<?php
// Set headers for CORS and JSON content
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: POST, GET, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type");
header("Content-Type: application/json; charset=UTF-8");

// Handle preflight OPTIONS request
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Include Database Connection
$conn = require_once 'db.php';

// Handle GET request (Fetch latest images)
if ($_SERVER['REQUEST_METHOD'] === 'GET') {
    try {
        // Fetch last 20 images including the Base64 data
        $result = $conn->query("SELECT id, name, type, size, data, created_at FROM uploads ORDER BY created_at DESC LIMIT 20");
        $images = $result->fetch_all(MYSQLI_ASSOC);
        echo json_encode($images);
    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode(["error" => "Failed to fetch images"]);
    }
    exit();
}

// Handle POST request (Upload images)
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Get JSON input
    $input = file_get_contents("php://input");
    $data = json_decode($input, true);

    if (!isset($data['images']) || !is_array($data['images']) || empty($data['images'])) {
        http_response_code(400);
        echo json_encode(["error" => "No images provided"]);
        exit();
    }

    try {
        $conn->begin_transaction();
        
        $stmt = $conn->prepare("INSERT INTO uploads (name, type, size, data) VALUES (?, ?, ?, ?)");
        
        $uploaded = [];
        foreach ($data['images'] as $img) {
            // Generate Random Name
            $ext = pathinfo($img['name'], PATHINFO_EXTENSION);
            $randomName = bin2hex(random_bytes(10)); // 20 random hex characters
            if ($ext) {
                $randomName .= '.' . $ext;
            }

            $stmt->bind_param("ssis", $randomName, $img['type'], $img['size'], $img['base64']);
            $stmt->execute();
            
            $uploaded[] = [
                'id' => $conn->insert_id,
                'name' => $randomName,
                'original_name' => $img['name']
            ];
        }
        
        $conn->commit();

        http_response_code(201);
        echo json_encode([
            "message" => "Images uploaded successfully to MySQL",
            "count" => count($uploaded),
            "files" => $uploaded
        ]);
    } catch (Exception $e) {
        $conn->rollback();
        http_response_code(500);
        echo json_encode(["error" => "Failed to upload images: " . $e->getMessage()]);
    }
    exit();
}

// 404 for other methods
http_response_code(404);
echo json_encode(["error" => "Endpoint not found"]);
?>
